/*******************************************************************************
 Tabwidth = 2

  name      : 3dview.cpp
  desc      : example program for the hugi12-3d-viewing-article
  author    : Christian Schlange  aka  tryx/Xography
  started   : Do, 08-27-1998
  last up   : Mi, 09-02-1998
  compiler  : wpp386.exe Watcom C++ v10.x-11.x
  comments  : This program displays a simple 3D-scene using the viewing
		  			  transformations described in the article. The source is just
		  			  illustrating how the view-transformation works, nothing more.
		  			  The source is rather optimized from the high-level-aspect,
		  			  and asm would speed up a LOT more (not speaking of the Qsort
		  			  depth sorter <g> and the matrix-vector multiplication, which
		  			  HAS to be done in asm for optimal speed). Just look at the
		  			  VCamera-class, that's the only interesting thing in this source.
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <string.h>
#include <math.h>
#include <i86.h>
#include <dos.h>

#ifdef   USE_SVGA_LIB
#include <poly.h>
#else
#include <graph.h>
#endif


//****************************************************************************
//
// DEFINES
//
//****************************************************************************


typedef unsigned char  byte;
typedef unsigned short word;
typedef unsigned int   dword;

#define PI 3.1415927
#define USE_UNION


//****************************************************************************
//
// VECTOR, MATRIX AND FACE-CLASSES
//
//****************************************************************************


class V3D // 3D-VECTOR-CLASS (USED FOR VERTICES)
{

 public:
 	#ifdef USE_UNION
 	union {									 // nameless union
	 	float E[3];						 // same memory adress but 2 different ways
	 	struct {							 // of accessing the elements - offers some
	 		float x,y,z;				 // advantages ( sizeof(V3D)=12 )
	 	};
	};
	#else
	float x,y,z;
	#endif

	void 				Show   			(void);
  void 				Set 				(float _x, float _y, float _z);
  float				Len	   	 		(void);
  void 				Normalize 	(void);
  void static Cross  	  	(V3D &a, V3D &b, V3D &c);
  void static Add       	(V3D &a, V3D &b, V3D &c);
  void			  Add         (V3D &a);
  void			  Sub         (V3D &a);
  void static ScalarMul 	(float s, V3D &a);
};

void V3D::Show (void) {
	printf ("(%8.5f, %8.5f, %8.5f )\n", x,y,z);
}

void V3D::Set (float _x, float _y, float _z) {
	x = _x;
	y = _y;
	z = _z;
}

float V3D::Len (void) {
	return (sqrt(x*x+y*y+z*z));
}

void V3D::Normalize (void) {
	float len = Len();
  if (len) { x /= len; y /= len; z /= len; }
}

void V3D::Cross (V3D &a, V3D &b, V3D &c) {
	c.x = a.y * b.z - a.z*b.y;
	c.y = a.z * b.x - a.x*b.z;
	c.z = a.x * b.y - a.y*b.x;
}

void V3D::Add (V3D &a, V3D &b, V3D &c) {	// see optimize.txt for an optimized
	c.x = a.x + b.x;												// version of this function (and for
	c.y = a.y + b.y;												// my opinion about modern compilers)
	c.z = a.z + b.z;
}

void V3D::Add (V3D &a) {
	x += a.x;
	y += a.y;
	z += a.z;
}
void V3D::Sub (V3D &a) {
	x -= a.x;
	y -= a.y;
	z -= a.z;
}

void V3D::ScalarMul (float s, V3D &a) {
	a.x *= s;
	a.y *= s;
	a.z *= s;
}


//----------------------------------------------------------------------------


class V2D // 2D-VECTOR CLASS
{
 public:
	float x,y;
};


//----------------------------------------------------------------------------


class FACE // FACE CLASS (HOLDING ONE 3DS-FACE)
{
 public:
	word  idx[3];
	byte  col,flags;
	V3D   normal;
};


// ---------------------------------------------------------------------------
// MATRIX CLASS
// 3x4 MATRIX, SAME LIKE 4x4-MATRIX WITH 4TH ROW 0 0 0 1
//
// AN IMPLEMENTATION WITHOUT CLASSES WOULD RESULT IN EXACTLY THE SAME SPEED
// AND SIZE.
// ---------------------------------------------------------------------------


class MAT34
{
 public:
  #ifdef  USE_UNION
 	union {
 		float E[3][4];							 // same memory adress but 2 different ways
 		struct {										 // of accessing the elements - quite nice
	 		float e00,e01,e02,e03;
	 		float e10,e11,e12,e13;
			float e20,e21,e22,e23;
		};
	};
	#else
	float E[3][4];
	#endif

	void Show    		 (void);
	void VecMul  		 (V3D &v, V3D &vd);
	void Clear   		 (void);
	void SetUnit 		 (void);
	void Mul         (MAT34 &a, MAT34 &b);
	void SetTranslation (float x, float y, float z);
	void SetScaling  (float sx, float sy, float sz);
	void SetRows     (V3D &r0, V3D &r1, V3D &r2);
	void SetRotation (float a, float b, float c);
	void SetVecRotation (float ux, float uy, float uz, float phi);

};

void MAT34::Show (void)
{
	int i,j;
	for ( i=0; i<3; i++ ) {
		for ( j=0; j<4; j++ ) printf ("%8.3f ", E[i][j]);
		printf ("\n");
	}
	printf ("\n");
}

void MAT34::Clear (void) {
	memset (E, 0, sizeof(E));
}

void MAT34::SetUnit (void) {
	this->Clear ();
	E[0][0] = E[1][1] = E[2][2] = 1.0;
}

// multiply a with b and store result in the calling object
void MAT34::Mul (MAT34 &a, MAT34 &b)
{
  int   i, j;
	float *aP, *bP;
	for (i=0; i<3; i++) {
	 	aP = a.E[i]; bP = b.E[0];
	 	for (j=0; j<4; j++, bP++) {
	 		E[i][j] = aP[0]*bP[0*4] + aP[1]*bP[1*4] + aP[2]*bP[2*4];
	 	}
	 	E[i][3] += aP[3];
	}
}

void MAT34::VecMul (V3D &v, V3D &vd) {
	vd.x = E[0][0]*v.x + E[0][1]*v.y + E[0][2]*v.z + E[0][3];
	vd.y = E[1][0]*v.x + E[1][1]*v.y + E[1][2]*v.z + E[1][3];
	vd.z = E[2][0]*v.x + E[2][1]*v.y + E[2][2]*v.z + E[2][3];
}

void MAT34::SetTranslation (float x, float y, float z)
{
	this->SetUnit ();
	E[0][3] = x;
	E[1][3] = y;
	E[2][3] = z;
}

void MAT34::SetScaling (float sx, float sy, float sz)
{
	this->SetUnit ();
	E[0][0] = sx;
	E[1][1] = sy;
	E[2][2] = sz;
}

void MAT34::SetRows (V3D &r0, V3D &r1, V3D &r2)
{
	this->Clear ();
	E[0][0] = r0.x; E[0][1] = r0.y; E[0][2] = r0.z;
	E[1][0] = r1.x; E[1][1] = r1.y; E[1][2] = r1.z;
	E[2][0] = r2.x; E[2][1] = r2.y; E[2][2] = r2.z;
}

// optimized version of the standard rotation-matrix for rotating a point
// about angle a,b,c in x,y,z direction (you might know this as the 9-mul
// rotation ;)
void MAT34::SetRotation (float a, float b, float c)
{
  float t2 = cos(b);
  float t3 = cos(a);
  float t5 = sin(a);
  float t7 = sin(b);
  float t8 = sin(c);
  float t9 = t8*t7;
  float t11 = cos(c);
  float t18 = t11*t7;
  E[0][0] = t2*t3;
  E[0][1] = -t2*t5;
  E[0][2] = t7;
  E[0][3] = 0.0;
  E[1][0] = t9*t3+t11*t5;
  E[1][1] = -t9*t5+t11*t3;
  E[1][2] = -t8*t2;
  E[1][3] = 0.0;
  E[2][0] = -t18*t3+t8*t5;
  E[2][1] = t18*t5+t8*t3;
  E[2][2] = t11*t2;
  E[2][3] = 0.0;
}

// optimized version for the rotation of angle phi around an arbitrary
// vector U (this matrix is a composition of 3 rotations: 1) rotate that
// U lies on z-axis 2) rotate with angle phi around z-axis 3) inverse
// rotation of point 1)

void MAT34::SetVecRotation (float ux, float uy, float uz, float phi)
{
	float t1 = ux*ux;
	float t2 = cos(phi);
	float t7 = 1.0-t2;
	float t8 = ux*uy*t7;
	float t9 = sin(phi);
	float t10 = uz*t9;
	float t13 = uz*ux*t7;
	float t14 = uy*t9;
	float t17 = uy*uy;
	float t22 = uy*uz*t7;
	float t23 = ux*t9;
	float t27 = uz*uz;
	E[0][0] = t1+t2*(1.0-t1);
	E[0][1] = t8-t10;
	E[0][2] = t13+t14;
	E[0][3] = 0.0;
	E[1][0] = t8+t10;
	E[1][1] = t17+t2*(1.0-t17);
	E[1][2] = t22-t23;
	E[1][3] = 0.0;
	E[2][0] = t13-t14;
	E[2][1] = t22+t23;
	E[2][2] = t27+t2*(1.0-t27);
	E[2][3] = 0.0;
}

// A non-class implementation could look like this :

#ifdef USE_NON_CLASS

	typedef float M34[3][4];

	void M34Clear (M34 &m) {
		memset (m, 0, sizeof(M34));
	}

	void M34SetUnit (M34 &m) {
		M34Clear (m);
		m[0][0] = m[1][1] = m[2][2] = 1.0;
	}

	void M34VecMul (M34 &m, V3D &v, V3D &vd) {
		vd.x = m[0][0]*v.x + m[0][1]*v.y + m[0][2]*v.z + m[0][3];
		vd.y = m[1][0]*v.x + m[1][1]*v.y + m[1][2]*v.z + m[1][3];
		vd.z = m[2][0]*v.x + m[2][1]*v.y + m[2][2]*v.z + m[2][3];
	}

	// and so on .....
#endif


//****************************************************************************
//
// VIRTUAL CAMERA
// Calculates View-Transformation Matrix
//
//****************************************************************************


class VCamera // VIRTUAL CAMERA
{
 public:
 	VCamera 	     (float _ures, float _vres, float _f, float _d, float _b);

 	void ViewTransform (void);
 	void TransformVCS (void);

 	void SetViewerPos(float x, float y, float z);
 	void SetRotation (float a, float b, float c);

 	float f,  d,  b;        	// distance front/VP/back
  float fp, dp, bp;					// s.a.o., but projected
	float angU, angV, angN; 	// rotation angles of VCS

	V3D   cw;                 // center of VP (in WCS-coordinates, pos of viewer)
	V3D   foc;		     				// focus position in VCS-coordinates (-d)
  V3D   vpn,vup,vri;       	// Viewplane normal and View-Up Vector
	V3D   vupR,vpnR,vriR;			// s.a.o, but rotated
  MAT34 matrix;	   					// the view-transformation matrix

 private:
	float ures, vres;	  			// dimensions of viewplane (width and height)
};


VCamera::VCamera (float _ures, float _vres, float _f, float _d, float _b) :
				  ures(_ures), vres(_vres), f(_f), d(_d), b(_b)
{
	cw.Set  (0,   0,-1000);		// some default value
	foc.Set (0,   0, -d);			// focus at pos (0,0,-d) on n-axis
	vpn.Set (0,   0, 1.0);		// set three vectors defining the VCS, VRI is
	vup.Set (0, 1.0,   0);		// calculated with VRI = VUP x VPN
	angU = angV = angN = 0;
}


void VCamera::SetViewerPos (float x, float y, float z) {
	cw.Set (x,y,z);
}

void VCamera::SetRotation (float a, float b, float c) {
	angU = a; angV = b; angN = c;
}


// Calculate the View-Transform ( in matrix VCamera::matrix )
//
// input: vriR,vupR,vpnR     -    three axis-vectors of the VCS
//
// vpn, vup, vpn are always used as a basis for transformation, and the
// result of this transformation is put to vriR, vupR and vpnR.

void VCamera::ViewTransform (void)
{
	MAT34 T1,R,T2,S1,m0,m1;

	vupR = vup;
	vpnR = vpn;

	V3D::Cross (vupR,vpnR,vriR);

	T1.SetTranslation (-cw.x,-cw.y,-cw.z); // setting up the 4 partial trans-
	R.SetRows         ( vriR, vupR, vpnR); // formations
	T2.SetTranslation (    0,    0,    d);


	S1.SetScaling (2/ures, 2/vres, 1/d );  // Scale1 : 2*d/ures, 2*d/vres, 1
																				 // Scale2 : 1/d

	m0.Mul     (  R, T1 );					// m0 = R * T1
	m1.Mul     ( S1, T2 );				  // m1 = S1 * T2
	matrix.Mul ( m1,m0 );						// MATRIX = S1 * T2 * R * T1

	fp = f / d;											// calculate the new position of the VP
	dp = 1;													// and Front/Back-clipping plane after
	bp = b / d;											// scaling the View Volume

}


// Here you can put transformations of the VCS-vectors (vup and vpn)
// The transformed vectors are input of VCamera::ViewTransform(void).
//
void VCamera::TransformVCS (void)
{
  MAT34 vcsRot;
	if (angU != 0) {
		vcsRot.SetVecRotation (vri.x,vri.y,vri.z,angU);
		vcsRot.VecMul(vup,vupR); vcsRot.VecMul(vpn,vpnR); vup=vupR;vpn=vpnR;
		angU = 0;
	}
  if (angV != 0) {
		vcsRot.SetVecRotation (vup.x,vup.y,vup.z,angV);
		vcsRot.VecMul(vpn,vpnR); vpn=vpnR;

		angV = 0;
	}
	if (angN != 0) {
		vcsRot.SetVecRotation (vpn.x,vpn.y,vpn.z,angN);
		vcsRot.VecMul(vup,vupR); vup=vupR;
		angN = 0;
	}
	vup.Normalize();
	vpn.Normalize();
	V3D::Cross (vup,vpn,vri);
}


//****************************************************************************
//
// 3D OBJECT CLASS ( MINI ASC-READER )
// optimized for size - go away and don't look at
// this piece of code, it's ugly and primitive
// if you want bigger objects, increase MAXMEM
// (didn't use dynamic allocaction for size reasons)
//
//****************************************************************************


#define MAXMEM 15000
#define SHADES 32

class Sort {
 public:
  word  idx;
  float meanZ;
};

class Object3D
{
 public:
  Object3D (char *_name);
 ~Object3D ( );
	void CalcNormals (void);

	int   vN,fN;		   	// # of vertices/faces
	float movestep;	   	// added to vcam-focus (depends on object dimensions)
	V3D   dim;			   	// span of object in the 3 dimensions

	V3D  *  V;			   	// original vertices
	V3D  *  VNormals;		// vertice normals
	V3D  *  VT;			   	// transformed vertices
	FACE *  F; 		   	  // faces
	Sort *  S;

};

Object3D::Object3D (char *_name)
{
	#define m 10E10
	FILE *IN;
	V3D  *vP=V=new V3D  [MAXMEM];
	FACE *fP=F=new FACE [MAXMEM];
	int   fbase, cbase=0,i;
	float min[3]={m,m,m},max[3]={-m,-m,-m},cen[3];
	char  line[128], *cP;
	vN = 0; fN = 0;
	movestep = 0;
	if (!(IN=fopen(_name, "r"))) exit(1);
	while (fgets(line,256,IN)) {
		if (strstr(line,"Vertex list:")) { fbase=vN; cbase++; }
		if (strstr(line,"Vertex") && (cP=strstr(line,"X:"))) {
			cP+=2; vP->x = atof(cP);
			cP = strstr(cP, "Y:")+2; vP->y=atof(cP);
			cP = strstr(cP, "Z:")+2; vP->z=atof(cP);
			if (min[0]>vP->x) min[0]=vP->x;if (min[1]>vP->y) min[1]=vP->y;if (min[2]>vP->z) min[2]=vP->z;
			if (max[0]<vP->x) max[0]=vP->x;if (max[1]<vP->y) max[1]=vP->y;if (max[2]<vP->z) max[2]=vP->z;
			vN++; vP++;
		}
		if (strstr(line,"Face") && (cP=strstr(line,"A:"))) {
			cP+=2; fP->idx[0] = atoi(cP)+fbase;
			cP = strstr(cP, "B:")+2; fP->idx[1] = atoi(cP)+fbase;
			cP = strstr(cP, "C:")+2; fP->idx[2] = atoi(cP)+fbase;
			fP->col=(cbase-1)*SHADES;
			fN++; fP++;
		}
	}
	fclose (IN);
	for (i=0; i<3; i++) {
		cen[i]=(min[i]+max[i])/2.0;
		if (max[i]-min[i]>movestep) movestep=max[i]-min[i];
	}
	for (vP=V,i=0; i<vN; i++, vP++) {
		vP->E[0] -= cen[0]; vP->E[1] -= cen[1]; vP->z -= cen[2];	// see above
	}
	VNormals = new V3D[vN];
	VT  		 = new V3D[vN];
	S   		 = new Sort[fN];
	CalcNormals ();
}

Object3D::~Object3D ()
{
	delete(V);
	delete(VT);
	delete(F);
}

void Object3D::CalcNormals (void)
{
	int   f, v;
	int   a, b, c, anz;
	float x1,y1,z1, x2,y2,z2, x3,y3,z3;
	float ux,uy,uz, vx,vy,vz, nx,ny,nz,nb;
	float mx,my,mz, mb;
	V3D  *normal;

  // Surface normals
  for ( f=0; f<fN; f++ )
	{
	    a = F[f].idx[0]; b = F[f].idx[1]; c = F[f].idx[2];
	    x1 = V[a].x; y1 = V[a].y; z1 = V[a].z;
	    x2 = V[b].x; y2 = V[b].y; z2 = V[b].z;
	    x3 = V[c].x; y3 = V[c].y; z3 = V[c].z;
	    ux = x2-x1;  uy = y2-y1;  uz =  z2-z1;
	    vx = x3-x1;  vy = y3-y1;  vz =  z3-z1;
	    nx = uy*vz - vy*uz;
	    ny = vx*uz - ux*vz;
	    nz = ux*vy - vx*uy;
	    nb = sqrt ( nx*nx + ny*ny + nz*nz );
	    if (nb > 0.0) {
	    	nx /= nb; ny /= nb; nz /= nb;
	    } else {
	    	nx = 1.0; ny = 0.0; nz = 0.0;
	    }
			F[f].normal.x = nx;
			F[f].normal.y = ny;
			F[f].normal.z = nz;
	}

	// Vertice normals (lame brute force)
	for (v=0; v<vN; v++)
	{
		anz = mx = my = mz = 0;
	  for (f=0; f<fN; f++)
	  {
			a = F[f].idx[0]; b = F[f].idx[1]; c = F[f].idx[2];
		  if (a==v || b==v || c==v) {
		     	anz ++;
		     	normal = &F[f].normal;
		     	mx += normal->x; my += normal->y; mz += normal->z;
		  }
	  }
	  if (anz != 0) {
		    mx /= float(anz); my /= float(anz); mz /= float(anz);
		}
		VNormals[v].Set(mx,my,mz);
	  VNormals[v].Normalize ();
		if (!(v%500)) { printf ("%i/%i ",v,vN); fflush(stdout); }
	}
}


//****************************************************************************
//
// MOUSE CLASS (VERY SIMPLE,NO HANDLER,JUST POLLING)
//
//****************************************************************************


class Mouse
{
 public:
  Mouse (void);
 ~Mouse () { Reset(); }
	int  MIrq  (void);
	int  Reset (void);
	int  Poll  (void);

	char   key;
	int    x, y;
  int    dx,dy;
  byte   lbutt, rbutt;
	byte   existing,d1;

 private:
  int    ox,oy;
  struct SREGS seg;
  union  REGS  in,out;

};

Mouse::Mouse (void) : existing(1),key(1)
{
	memset(&in,0,sizeof(REGS));
	if (!Reset()) {
    printf( "mouse not found, using keyboard...\n" ); exit (1);
  }
}

int Mouse::MIrq (void) {
	return (int386 (0x33, &in, &out));
}

int Mouse::Reset (void) {
  in.w.ax=0; MIrq();
  return  (out.w.ax);
}

int Mouse::Poll (void) {
  int res=0;
	in.w.ax = 0xB; MIrq();
	dx  = short(out.w.cx);
	dy  = short(out.w.dx);
	in.w.ax = 0x3;MIrq();
	x = out.w.cx; y = out.w.dx;
	lbutt = (out.w.bx&1);
	rbutt = (out.w.bx&2);
	if ( key==1 ) res = 1;
	key = 0; if (kbhit()) key = getch();
	return (dx|dy|lbutt|rbutt|key);
}


//****************************************************************************
//
// RENDER CLASS (INCLUDES TRANSFORM, FAKE CLIPPING AND DISPLAY)
//
//****************************************************************************

class Render
{
  public:
  	Render (Object3D &_obj, VCamera &_vcam, int _xres, int _yres);
  	RenderScene (void);
		void SetPal (void);
		void QSort (Sort a[], int l, int r);

  private:
    int      rendermode, culling;
    float    xres, yres;

    Object3D &obj;
    VCamera  &vcam;
};

Render::Render (Object3D &_obj, VCamera &_vcam, int _xres, int _yres) :
							 obj(_obj), vcam(_vcam), xres(_xres/2), yres(_yres/2)
{
	rendermode = 2;
	culling		 = 1;
}

void Render::QSort (Sort a[], int l, int r)
{
	int   i,j;
	float z;
	Sort  t;

	if (r>l)
	{
	  z = a[r].meanZ; i = l-1; j = r;
	  for (;;)
	  {
	    do i++; while (a[i].meanZ < z);
	    do j--; while (a[j].meanZ > z);
	    if (i>=j) break;
	    t=a[i]; a[i]=a[j]; a[j]=t;
	  }
	  t=a[i]; a[i]=a[r]; a[r]=t;
	  QSort (a,l,i-1);
	  QSort (a,i+1,r);
   }
}


void _SetPal (int col, byte r, byte g, byte b);
#pragma aux _SetPal = \
	"mov dx, 03c8H"\
	"out dx, al"\
	"inc dx"\
	"mov al, bl"\
	"out dx, al"\
	"mov al, bh"\
	"out dx, al"\
	"mov al, cl"\
	"out dx, al"\
	parm [eax] [bl] [bh] [cl] \
	modify [eax edx];

void Render::SetPal (void) {
	int  s, i;
	byte g[256/SHADES][3]={0,1,2, 1,2,1, 2,0,1, 1,2,2, 2,1,2, 2,2,1, 1,1,2, 0,0,2};
	for (s=0; s<256/SHADES; s++) {
		for (i=0; i<SHADES; i++) {
			_SetPal (s * SHADES + i,g[s][0]*i,g[s][1]*i,g[s][2]*i); //slow
		}
	}
}

Render::RenderScene (void)
{

	word * I;

	int    vis;
	int    v, f;
	int    a,b,c;
	int    shade;
	int    s1,s2,s3;
	int    x1,y1,x2,y2,x3,y3;

	float  dist;
	float  z1,z2,z3;
 	float  speedZ=0;

	V3D  * vP, *vtP;
	FACE * fP;
	Sort * sP;

	SetPal ();
	Mouse  cheese;

	// MAIN LOOP
	do
	{

		vcam.TransformVCS  ();							// move camera (put own code here)
		vcam.ViewTransform ();							// calculate the viewing transformation
																				// matrix dependend on camera

		// TRANSFORM VERTICES

    vP   = obj.V;
		vtP  = obj.VT;
		dist = vcam.dp;
		v    = obj.vN;
		do
		{
			vcam.matrix.VecMul (*vP,*vtP);
			z1      = dist / vtP->z;
			vtP->x  = vtP->x * z1 * xres + xres;
			vtP->y  = yres - vtP->y * z1 * yres;
			vP++; vtP++;
		} while (--v);

		// SORT FACES	(USING QSORT <- NO SPACE)

		vtP  = obj.VT;
		fP   = obj.F;
		sP   = obj.S;

	  for ( f=0; f<obj.fN; f++, fP++, sP++ ) {
		 	I = fP->idx;
		 	a = I[0];// b = I[1]; c = I[2];
			sP->idx   = f;
			sP->meanZ = vtP[a].z;// + vtP[b].z + vtP[c].z;
		}

		QSort (obj.S, 0, obj.fN-1);

		// RENDER THE OBJECT

		vtP  = obj.VT;
		fP   = obj.F;
		sP   = obj.S;

		#ifdef USE_SVGA_LIB
		SwapPages   (0);
		ClearScreen ( );
		#else
		_clearscreen (_GCLEARSCREEN);
		#endif

	  for (f=obj.fN-1; f>=0; f-- )
	  {

		 	fP = &obj.F[sP[f].idx];
		 	word * I = fP->idx;

			a=I[0]; b=I[1]; c=I[2];

			x1=vtP[a].x, y1=vtP[a].y; z1=vtP[a].z;
			x2=vtP[b].x, y2=vtP[b].y; z2=vtP[b].z;
			x3=vtP[c].x, y3=vtP[c].y; z3=vtP[c].z;

		 	// Do fake 3D-clipping
		 	if ( z1<vcam.bp && z1>vcam.fp && z2>vcam.fp && z3>vcam.fp )
		 	{

				if (culling) vis = (x2-x1)*(y3-y2)-(x3-x2)*(y2-y1);
				else vis = 1;

				if (vis > 0)
				{
					z1 = fP->normal.z;
					s1 = int(SHADES/2 - z1*float(SHADES)/3);
					s1 += fP->col;

					#ifdef USE_SVGA_LIB

					int flat[6];
					int gouraud[9];

					if (rendermode==0) {
						flat[0] = x1; flat[1] = y1;
						flat[2] = x2; flat[3] = y2;
						flat[4] = x3; flat[5] = y3;
						Poly_FLAT (flat, s1);
					} else {
					  V3D * vnP=obj.VNormals;
						z1 = vnP[a].z; z2 = vnP[b].z; z3 = vnP[c].z;
						s1 = int(SHADES/2 - z1*float(SHADES)/3);
						s2 = int(SHADES/2 - z2*float(SHADES)/3);
						s3 = int(SHADES/2 - z3*float(SHADES)/3);
						s1 += fP->col; s2 += fP->col; s3 += fP->col;
						gouraud[0] = x1; gouraud[1] = y1; gouraud[2] = s1;
						gouraud[3] = x2; gouraud[4] = y2; gouraud[5] = s2;
						gouraud[6] = x3; gouraud[7] = y3; gouraud[8] = s3;
						Poly_GS (gouraud);
					}

					#else
					struct xycoord points[3];
					_setcolor ( s1 );
					if (rendermode==0)
						_setpixel (x1,y1);
					else if (rendermode==1) {
						_moveto (x1,y1);
						_lineto (x2,y2);
						_lineto (x3,y3);
						_lineto (x1,y1);		// face-flags would speed up
					} else if (rendermode==2) {
					 	points[0].xcoord = x1; points[0].ycoord = y1;
					 	points[1].xcoord = x2; points[1].ycoord = y2;
					 	points[2].xcoord = x3; points[2].ycoord = y3;
					 	_polygon  ( _GFILLINTERIOR, 3, points);
					}

					#endif
				}

			}

		}

  	// MOVEMENT

   	#ifdef USE_SVGA_LIB
   	cheese.Poll ();
   	#else
   	while (!cheese.Poll ());
   	#endif

	  float addX,addY,addZ;
	  V3D   panX = vcam.vriR;
	  V3D   panY = vcam.vupR;
	  V3D   panZ = vcam.vpnR;
	  float panA = obj.movestep/500.0;

   	if (cheese.lbutt && cheese.rbutt)
   	{
	   	vcam.angN = -float(cheese.dx)/500.0;
   	}
   	else if (cheese.lbutt)
   	{
   		addX=float(cheese.dx)*panA*.2; V3D::ScalarMul (addX,panX); vcam.cw.Add(panX);
   		addY=float(cheese.dy)*panA*.2; V3D::ScalarMul (addY,panY); vcam.cw.Add(panY);
   	}
   	 else	if (cheese.rbutt)
   	{
   		addZ=float(cheese.dy)*panA*.2; V3D::ScalarMul(addZ,panZ); vcam.cw.Sub(panZ);
   	}
   	 else
   	{
	   	vcam.angU = -float(cheese.dy)/3000.0;
	   	vcam.angV = float(cheese.dx)/3000.0;
	  }

		V3D::ScalarMul(speedZ,panZ); vcam.cw.Sub(panZ);

		if (cheese.key) {
			switch (cheese.key) {
				case '8' : speedZ -= panA; break;
				case '2' : speedZ += panA; break;
				case '5' : speedZ  = 0;	break;
				case '1' : V3D::ScalarMul (-1,vcam.vpn); break;
				case 0x20: rendermode = (rendermode+1)%3; break;
				case '<' : culling ^= 1; break;
				default  : break;
			}
		}

	} while (cheese.key!=27);

}


//****************************************************************************
//
// MAIN	& INFO
//
//****************************************************************************


#pragma aux _setvmode = \
	"int  10h"\
	parm [eax];

void InfoScreen (void)
{
	printf ("Control Info:\n\n"
					"mouse:\n\n"
					"move mouse          : rotate\n"
					"left button + move  : strafe (pan)\n"
					"right button + move : move forward/back\n"
					"both buttons + move : roll camera\n\n"
					"keyboard:\n\n"
					"2/8                 : acceleration control\n"
					"5                   : stop movement\n"
					"1                   : turn back/forward\n\n"
					"space               : toggle rendering mode\n"
					"<                   : toggle backface culling\n"
					"escape              : exit\n\n"
					"simple rotations of the VCS are used to simulate\n"
					"camera movements. Use a more complex movement con-\n"
					"trol instead. Only Fake-3D-clipping is supported, and\n"
					"the camera movement isn't very exact (produces errors\n"
					"because of the rotation method (rotation about an arbi\n"
					"trary axis\n"
					"<key>\n");
	getch ();
}

void main (int argc, char *argv[])
{

	float  aspect;
	int    xres, yres;

	if (argc<2) {
		printf ("syntax: 3DVIEW.EXE [object.asc]\n");
		exit   (1);
	}

	// GRAPHICS SETUP
	#ifndef USE_SVGA_LIB
		struct videoconfig vc;
		if ( !_setvideomode (_VRES256COLOR) ) {
		  printf ("Could not initialize gfx-mode\n");
		  exit   (1);
		}
		_getvideoconfig (&vc);
		xres = vc.numxpixels;
		yres = vc.numypixels;
	#else
		xres = 640; yres = 480;			 				// it does only work for this res
		InitSVGAAsmLib (xres, yres);
	#endif
	aspect = float(yres) / float(xres);

	InfoScreen ();


	// LOAD OBJECT

	Object3D Obj      ( argv[1] );


	/*
	INIT VIRTUAL CAMERA
	(1,aspect) - viewplane dimensions, (1,1) would distort the scene
	the next 3 parameters are the distances of the front-clipping-plane,
	viewplane and back-clipping-plane (distance on the pos. n-axis (that
	is the z-axis of the VCS) from the VCS-origin
	*/

	VCamera  VCam     ( 1,aspect, 0.5,2.0,20000 );


	// SET VIEWER POSITION (dependant on object-size)

	VCam.SetViewerPos ( 0,0,-Obj.movestep*1.5 );


	// SETUP RENDER CLASS AND DO THE RENDERING

	Render   Rend     ( Obj, VCam, xres, yres );
	Rend.RenderScene  ( );


	_setvmode (3);
}

